﻿//---------------------------------------------------------------------------------
// Microsoft (R) .NET Services SDK
// Software Development Kit
// 
// Copyright (c) Microsoft Corporation. All rights reserved.  
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE. 
//---------------------------------------------------------------------------------

namespace Microsoft.AccessControl.ASPNET
{
    using System;
    using System.Collections.Generic;
    using System.Configuration;
    using System.Linq;
    using System.Security.Principal;
    using System.Text;
    using System.Web.Configuration;

    public class AuthorizationManager
    {
        private CommaDelimitedStringCollection allowedRoles = new CommaDelimitedStringCollection();
        private CommaDelimitedStringCollection deniedRoles = new CommaDelimitedStringCollection();
        private CommaDelimitedStringCollection allowedUsers = new CommaDelimitedStringCollection();
        private CommaDelimitedStringCollection deniedUsers = new CommaDelimitedStringCollection();

        private AuthorizationManager()
        {
        }

        public CommaDelimitedStringCollection AllowedRoles
        {
            get
            {
                return this.allowedRoles;
            }
        }

        public CommaDelimitedStringCollection DeniedRoles
        {
            get
            {
                return this.deniedRoles;
            }
        }

        public CommaDelimitedStringCollection AllowedUsers
        {
            get
            {
                return this.allowedUsers;
            }
        }

        public CommaDelimitedStringCollection DeniedUsers
        {
            get
            {
                return this.deniedUsers;
            }
        }

        public bool HasRoles
        {
            get
            {
                return this.AllowedRoles.Count > 0 || this.DeniedRoles.Count > 0;
            }
        }

        public bool HasExplicitUsers
        {
            get
            {
                bool hasUsers = false;

                if (this.AllowedUsers.Count == 1 &&
                    this.AllowedUsers.Contains("*") &&
                    this.AllowedRoles.Count == 0)
                {
                    // all users are implicitly allowed (*)
                }
                else
                {
                    if ((this.AllowedUsers.Count > 0) || (this.DeniedUsers.Count > 0))
                    {
                        hasUsers = true;
                    }
                }

                return hasUsers;
            }
        }

        public bool RequiresAuthorization
        {
            get
            {
                return this.HasRoles || this.HasExplicitUsers;
            }
        }

        public static AuthorizationManager CreateAuthorizationManager(string url)
        {
            Configuration config = WebConfigurationManager.OpenWebConfiguration(url);

            AuthorizationSection configSection = (AuthorizationSection)config.GetSection("system.web/authorization");

            AuthorizationRuleCollection rules = configSection.Rules;

            AuthorizationManager authManager = new AuthorizationManager();

            for (int i = 0; i < rules.Count; i++)
            {
                if (rules[i].Roles.Count > 0)
                {
                    if (rules[i].Action.ToString() == "Allow")
                    {
                        authManager.AllowedRoles.AddRange(rules[i].Roles.ToString().Split(','));
                    }
                    else if (rules[i].Action.ToString() == "Deny")
                    {
                        authManager.DeniedRoles.AddRange(rules[i].Roles.ToString().Split(','));
                    }
                }

                if (rules[i].Users.Count > 0)
                {
                    if (rules[i].Action.ToString() == "Allow")
                    {
                        authManager.AllowedUsers.AddRange(rules[i].Users.ToString().Split(','));
                    }
                    else if (rules[i].Action.ToString() == "Deny")
                    {
                        authManager.DeniedUsers.AddRange(rules[i].Users.ToString().Split(','));
                    }
                }
            }

            return authManager;
        }

        public bool VerifyAuthorization(IPrincipal user)
        {
            // if the user is explicitly denied access, return false
            if (this.DeniedUsers.Contains(user.Identity.Name))
            {
                return false;
            }

            // if the user is explicitly allowed access, return true
            if (this.AllowedUsers.Contains(user.Identity.Name))
            {
                return true;
            }

            // check if user is denied by role
            bool userDeniedByRole = false;
            foreach (string role in this.DeniedRoles)
            {
                if (user.IsInRole(role))
                {
                    userDeniedByRole = true;
                    break;
                }
            }

            // if the user is denied explicitly by role, return false
            if (userDeniedByRole)
            {
                return false;
            }

            // check if the user is allowed by role
            bool userAllowedByRole = false;
            foreach (string role in this.AllowedRoles)
            {
                if (user.IsInRole(role))
                {
                    userAllowedByRole = true;
                    break;
                }
            }

            // if the user is allowed by role, return true
            if (userAllowedByRole)
            {
                return true;
            }

            // if all users are denied, return false
            if (this.DeniedUsers.Contains("*"))
            {
                return false;
            }

            // if all roles are denied, return false
            if (this.DeniedRoles.Contains("*"))
            {
                return false;
            }

            // otherwise, return true
            return false;
        }
    }
}
